% 2D SLAM with linear KF - Moving vehicle - Relative measurement - Limited
% Sensor Range - Observing (x,v,landmarks) - Landmarks updated once
% observed
%
% A 2 DOF mobile robot is moving along a path detecting some
% motionless landmarks; the position/velocity of the robot and position
% of landmarks are computed by means of Simultaneos Localization and 
% Mapping using a linear Kalman Filter.  
% All the measures are relative to robot position.
% The measuring sensor is limited to a certain range distance.
% Landmarks are updated in the state once observed.
%
% Programmed by Joaquim Salvi

clear all;
NumberTimeStamps = 600;
MapDimension = [1,100;1,100];
MaximumRange = 20;

% INTRODUCE LANDMARKS
figure(1); clf; title('Introduce Landmarks');
v=[MapDimension(1,1) MapDimension(1,2) MapDimension(2,1) MapDimension(2,2)];
axis(v); hold on;
pin=1; button = 0;
nlandmarks=0;
% Get input landmarks graphically, click right button to finish
while button ~= 3
   [x,y,button]=ginput(1);
   pin= ~isempty(x);
   if pin && button ~= 3
      nlandmarks = nlandmarks+1;
      plot(x,y,'r*')
      pp(:,nlandmarks)=[x;y];
   end
end
hold off;
NumberLandmarks = size(pp,2);

% DEFINE TRAJECTORY
% Definition of the real trajectory; vehicle with constant velocity
figure(2); clf;
title('Introduce trajectory around landmarks');
v=[MapDimension(1,1) MapDimension(1,2) MapDimension(2,1) MapDimension(2,2)];
axis(v); hold on;
plot(pp(1,:),pp(2,:),'r*');
pin = 1; button = 0; fi = 0;
npoints = 0; dist = 0;
% get trajectory segments, click right button to finish 
while button ~= 3
   [x,y,button]=ginput(1);
   pin= ~isempty(x);
   if pin && button ~= 3
      npoints = npoints + 1;
      t(:,npoints)=[x;y];
      plot(x,y,'go');
      if npoints > 1
        plot(t(1,npoints-1:npoints),t(2,npoints-1:npoints),'g-');
        dist = dist + norm(t(:,npoints) - t(:,npoints-1));
      end
   end
end
% Sampling NumberTimeStamps points in the given trajectory.
point = 2; dist2=0; incdist=dist/NumberTimeStamps;
tt(:,1)=t(:,1);
for i = 2:NumberTimeStamps
    tt(:,i)=tt(:,i-1)+ incdist*((t(:,point)-t(:,point-1))/norm(t(:,point)-t(:,point-1))); % tx,ty trajectories
    vv(:,i-1)=tt(:,i)-tt(:,i-1); % vx,vy velocities
    plot(tt(1,i),tt(2,i),'b.');
    dist2 = dist2 + incdist;
    if (dist2 + incdist) > norm(t(:,point)-t(:,point-1)) && abs((dist2 + incdist)-norm(t(:,point)-t(:,point-1))) > abs(dist2-norm(t(:,point)-t(:,point-1)))
        point = point + 1; dist2 = 0;
    end
end
plot(tt(1,:),tt(2,:),'b.');
hold off;

% DEFINE INITIAL PARAMETERS
v = vv;  % Velocity norm is constant and equal to 1, but not vx, vy
x = tt;  % Trajectory is the interpolated trajectory introduced by user
         % Landmarks are arranged in a vector form (x1,y1,x2,y2,... yn)
clear vv;
for i=1:NumberLandmarks
    p(2*i-1)=pp(1,i);
    p(2*i)=pp(2,i);
end     

% Definition of process noise and initial noises
r = (30*0.5)^2;  % landmark measurement noise, change to see effect
rx = (3*0.5)^2; % position measurement noise, change to see effect
rv = (0.1*0.5)^2; % velocity measurement noise, change to see effect
pn = (0.001)^2; % process noise, change to see effect

% Definition of Measurement matrix to use in prediction
% Landmark distance are relative to robot position
HP = diag(ones(2+2,1));

% Definition of Measurement matrix to use in ground truth
% Landmark distance are relative to robot position
HO = [diag(ones(2+2,1)) zeros(2+2,2*NumberLandmarks)];
for i=1:2:2*NumberLandmarks
    HO = [HO;-1 0 0 0 zeros(1,i-1) 1 zeros(1,2*NumberLandmarks-i)];
    HO = [HO;0 -1 0 0 zeros(1,i) 1 zeros(1,2*NumberLandmarks-i-1)];
end

% Definition of the State matrix
% The State matrix is a diagonal of ones, which means that the prediction
% of next state is equivalent to current state, except the position which
% is updated with the current velocity at every time stamp.
Fk = diag(ones(1,2+2)); Fk(1,3)=1; Fk(2,4)=1;   

% Definition of the Control vector
% The control vector is null, which means that the are no external inputs
% that change the robot state.
Uk = zeros(2+2,1);  

% Definition of the Covariance matrix
% The initial covariance matrix is defined with a large diagonal uncertainity
% in both state of the robot and position of the landmarks and with an
% equivalent uncertainity which means that none prevail over the other.
Pk = [(rx+rv) 0 rv 0;0 (rx+rv) 0 rv;rv 0 rv 0;0 rv 0 rv];

%Definition of the Process Noise Matrix
Q = diag(pn*ones(1,2+2));
%Definition of the Measurement Noise Matrix
R = diag([rx*ones(1,2) rv*ones(1,2)]); 

% Initial prediction of the vector state
for kk = 1:NumberTimeStamps
    xhat{kk} = [];
    zhat{kk} = [];
    Un{kk} = [];
end
xhat{1}= [x(:,1)' v(:,1)']';
N = sqrt(pn)*randn(2+2,1); xhat{1}(1:4) = xhat{1}(1:4) + N; % adding process noise
xtt(:,1) = xhat{1}(1:2);  % Only for plotting purposes

% Definition of measurement noises (may differ from initial noises
r = (20*0.5)^2;  % landmark measurement noise, change to see effect
rx = (1*0.5)^2; % position measurement noise, change to see effect
rv = (0.1*0.5)^2; % velocity measurement noise, change to see effect

% KALMAN FILTER LOOP
figure(3); clf; 
handle = gcf;
set(handle,'DoubleBuffer','on')

landobs = []; % Set of landmarks observed at least once, i.e. already updated in Pk.
plotland(:,1) = zeros(NumberLandmarks,1);
for k = 1:NumberTimeStamps-2
    
    % Prediction
    xhat{k+1} = Fk*xhat{k}+Uk;     % Prediction of next state
    zhat{k+1} = HP*xhat{k+1};      % Measure at the predicted position
    Pk = Fk*Pk*Fk' + Q;
    
    % Observation
    N = [sqrt(rx)*randn(2,1); sqrt(rv)*randn(2,1); sqrt(r)*randn(2*NumberLandmarks,1)];
    z(:,k+1)= HO*[x(:,k+1)' v(:,k+1)' p]' + N; % Measure at the correct position x(k+1) v(k+1) and p
    vvt=z(1:4,k+1)-zhat{k+1}(1:4);  % Innovation vector, i.e. discrepancy between measures
    znot(:,k+1)= HO*[x(:,k+1)' v(:,k+1)' p]'; % Measure at the correct position x(k+1) v(k+1) and p
    
    landdec = []; landnotdec = []; % landmarks detected/non-detected sets
    plotland(:,k+1) = zeros(NumberLandmarks,1); % only for plotting purposes
    for kk=5:2:5+2*NumberLandmarks-1  % Explore the detection of landmarks
        if norm(znot(kk:kk+1,k+1)) < MaximumRange % Landmark in not-noisy range?       
            landdec = [landdec; (kk-1)/2-1]; % Add detected landmark identifier to the set
            [check,pos] = ismember((kk-1)/2-1,landobs); % Check if landmark was observed in the past
            if check
                vvt=[vvt;z(kk:kk+1,k+1)-zhat{k+1}(4+2*pos-1:4+2*pos)];  % Innovation vector, i.e. discrepancy between measures
                plotland((kk-1)/2-1,k+1) = 1;
            else
                landnotdec = [landnotdec; (kk-1)/2-1]; % Add landmark as non-detected for another step
            end
        else
            landnotdec = [landnotdec; (kk-1)/2-1]; % Add non-detected landmark identifier to the set
        end
    end
    
    Ht = HP; 
    for kk=size(landobs,1):-1:1  % Removing rows/column of once observed but now non-detected landmarks
        [check,pos] = ismember(landobs(kk),landnotdec);
        if check
            Ht(4+2*kk-1:4+2*kk,:) = []; 
        end
    end

    Rt = diag([rx*ones(1,2) rv*ones(1,2) r*ones(1,size(Ht,1)-4)]);
    S = Ht*Pk*Ht' + Rt;
    
    % Update
    W = Pk * Ht' * inv(S); 
    xhat{k+1} = xhat{k+1}+W*vvt;
    Pk = Pk - W*Ht*Pk;

    Un{k+1}=diag(Pk);
    xtt(:,k+1) = xhat{k+1}(1:2);
    
    % Plotting results
    clf; hold on;
    title('2D SLAM with KF');
    mapdim=[MapDimension(1,1) MapDimension(1,2) MapDimension(2,1) MapDimension(2,2)];
    axis(mapdim); 
    plot(x(1,1:k),x(2,1:k),'g');
    plot(xtt(1,1:k),xtt(2,1:k),'r');
    plot(x(1,k+1),x(2,k+1),'g.');
    plot(xhat{k+1}(1),xhat{k+1}(2),'r.');
    [xe,ye,ze]=ellipsoid(xhat{k+1}(1),xhat{k+1}(2),0,3*sqrt(Un{k+1}(1)),3*sqrt(Un{k+1}(2)),0);
    plot(xe(floor(size(xe,2)/2),:),ye(floor(size(ye,2)/2),:));
    plot(pp(1,:),pp(2,:),'r*');
    for j=5:2:size(xhat{k+1},1)-1
%        plot(xhat(1,1:k+1)+zhat(j,1:k+1),xhat(2,1:k+1)+zhat(j+1,1:k+1),'m.');
        plot(xhat{k+1}(j),xhat{k+1}(j+1),'b.');
        [xe,ye,ze]=ellipsoid(xhat{k+1}(j),xhat{k+1}(j+1),0,3*sqrt(Un{k+1}(j)),3*sqrt(Un{k+1}(j+1)),0);
        plot(xe(floor(size(xe,2)/2),:),ye(floor(size(ye,2)/2),:));
    end
    drawnow;
    hold off;

    % Adding detected landmarks to the state
    for kk=1:size(landdec,1)
        if ~ismember(landdec(kk),landobs)
            landobs = [landobs; landdec(kk)];
            landobs = sort(landobs,'ascend');
            [check,pos] = ismember(landdec(kk),landobs);
            if pos == size(landobs,1)
                Fk = [Fk zeros(size(Fk,1),2)]; Fk = [Fk; [zeros(2,size(Fk,2)-2) [1 0; 0 1]]];
                Q = [Q zeros(size(Q,1),2)]; Q = [Q; [zeros(2,size(Q,2)-2) [pn 0; 0 pn]]];
                Pk = [Pk Pk(1:size(Pk,1),1:2)]; Pk = [Pk; [Pk(1:2,1:size(Pk,2)-2) [Pk(1,1)+r 0; 0 Pk(1,1)+r]]];
                xhat{k+1} = [xhat{k+1}; xhat{k+1}(1:2) + z(4+2*landdec(kk)-1:4+2*landdec(kk),k+1)];
%                xhat{k+1} = [xhat{k+1}; p(2*landdec(kk)-1:2*landdec(kk))']; 
                Uk = [Uk; zeros(2,1)];
                HP = [HP zeros(2,size(HP,1))']; HP = [HP; [[-1 0; 0 -1] zeros(2,size(HP,2)-4) [1 0; 0 1]]];
            else
                Fk = [Fk(:,1:4+2*(pos-1)) zeros(size(Fk,1),2) Fk(:,4+2*(pos-1)+1:size(Fk,2))]; 
                Fk = [Fk(1:4+2*(pos-1),:); zeros(2,size(Fk,2)); Fk(4+2*(pos-1)+1:size(Fk,1),:)];
                Fk(4+2*pos-1,4+2*pos-1) = 1; Fk(4+2*pos,4+2*pos) = 1;
                Q = [Q(:,1:4+2*(pos-1)) zeros(size(Q,1),2) Q(:,4+2*(pos-1)+1:size(Q,2))]; 
                Q = [Q(1:4+2*(pos-1),:); zeros(2,size(Q,2)); Q(4+2*(pos-1)+1:size(Q,1),:)];
                Q(4+2*pos-1,4+2*pos-1) = pn; Q(4+2*pos,4+2*pos) = pn;
                Pk = [Pk(:,1:4+2*(pos-1)) Pk(1:size(Pk,1),1:2) Pk(:,4+2*(pos-1)+1:size(Pk,2))]; 
                Pk = [Pk(1:4+2*(pos-1),:); Pk(1:2,1:size(Pk,2)); Pk(4+2*(pos-1)+1:size(Pk,1),:)];
                Pk(4+2*pos-1,4+2*pos-1) = Pk(1,1)+r; Pk(4+2*pos,4+2*pos) = Pk(2,2)+r;
                xhat{k+1} = [xhat{k+1}(1:4+2*(pos-1)); xhat{k+1}(1:2) + z(4+2*landdec(kk)-1:4+2*landdec(kk),k+1); xhat{k+1}(4+2*(pos-1)+1:size(xhat{k+1},1))];
%                xhat{k+1} = [xhat{k+1}(1:4+2*(pos-1)); p(2*landdec(kk)-1:2*landdec(kk))'; xhat{k+1}(4+2*(pos-1)+1:size(xhat{k+1},1))];
                Uk = [Uk(1:4+2*(pos-1)); zeros(2,1); Uk(4+2*(pos-1)+1:size(Uk,1))];
                HP = [HP(:,1:4+2*(pos-1)) zeros(size(HP,1),2) HP(:,4+2*(pos-1)+1:size(HP,2))]; 
                HP = [HP(1:4+2*(pos-1),:); [[-1 0; 0 -1] zeros(2,2*pos) [1 0; 0 1] zeros(2,size(HP,2)-4-2*pos)]; HP(4+2*(pos-1)+1:size(HP,1),:)];
            end
        end
    end
    
end;

% PLOTTING RESULTS
figure(4); clf; title('2D SLAM with KF');
mapdim=[MapDimension(1,1) MapDimension(1,2) MapDimension(2,1) MapDimension(2,2)];
axis(mapdim); hold on;
for i = 2:NumberTimeStamps-2
   plot(x(1,i),x(2,i),'g.');
   plot(xhat{i}(1),xhat{i}(2),'r.');
   [xe,ye,ze]=ellipsoid(xhat{i}(1),xhat{i}(2),0,3*sqrt(Un{i}(1)),3*sqrt(Un{i}(2)),0);
   plot(xe(floor(size(xe,2)/2),:),ye(floor(size(ye,2)/2),:));
   kk = 1;
   for j=1:NumberLandmarks
       if max(plotland(j,1:i)) == 1
           plot(xhat{i}(1)+zhat{i}(4+kk),xhat{i}(2)+zhat{i}(4+kk+1),'m.');
           plot(xhat{i}(4+kk),xhat{i}(4+kk+1),'b.'); 
           if mod(i,10) == 2
               [xe,ye,ze]=ellipsoid(xhat{i}(4+kk),xhat{i}(4+kk+1),0,3*sqrt(Un{i}(4+kk)),3*sqrt(Un{i}(4+kk+1)),0);
               plot(xe(floor(size(xe,2)/2),:),ye(floor(size(ye,2)/2),:));
           end
           kk = kk + 2;
       end
   end
end
kk = 1;
for j=1:NumberLandmarks
    plot(pp(1,j),pp(2,j),'r*');
    st=sprintf('%d',j); 
    text(pp(1,j)+2,pp(2,j)+2,st);
    if max(plotland(j,1:NumberTimeStamps-2)) == 1
        plot(xhat{NumberTimeStamps-2}(4+kk),xhat{NumberTimeStamps-2}(4+kk+1),'go');
        kk = kk + 2;
    end
end
hold off; zoom on;

